/* gpgez_decrypt.c
 * Copyright (C) 2005 Kazuyoshi Kakihara
 *
 * This file is part of JanusDG.
 *
 * JanusDG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <windows.h>

#include "gpgez.h"
#include "gpgez_util.h"
#include "gpgez_rungpg.h"

#define STATUS_VALIDSIGN 1
#define STATUS_END_DECRYPTION 2
#define STATUS_DECRYPTION_FAILED 4

#define GPGMSG_VALIDSIG "VALIDSIG"
#define GPGMSG_DECRYPTION_FAILED "DECRYPTION_FAILED"
#define GPGMSG_END_DECRYPTION "END_DECRYPTION"
#define GPGMSG_BAD_PASSPHRASE "BAD_PASSPHRASE"

struct gpgez_decrypt_params {
  LPDWORD lpdwStatus;
  LPCSTR szPassphrase;
};

DWORD _gpgez_dwProcDecrypt(struct childprocess *pcp, LPVOID pBuff);

/*------------------------------------------------------------*/
DWORD
gpgez_decrypt(LPCSTR szGpgHome, LPCSTR szSecRing, LPCSTR szPubRing, LPCSTR szFilePlain, LPCSTR szFileEncrypted, LPCSTR szPassphrase, LPDWORD lpdwStatus) {
  CHAR szCommandLineOptions[MAX_PATH * 6];
  struct gpgez_decrypt_params gdp;
  DWORD dwErr;
  
  if ((dwErr = gpgez_dwIsValidGpgHome(szGpgHome)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsExistSecRing(szSecRing)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsExistPubRing(szPubRing)) != ERR_NO_ERROR) return dwErr;
  
  if (_snprintf(szCommandLineOptions, sizeof(szCommandLineOptions), 
          "%s --homedir=\"%s\" --secret-keyring=\"%s\" --keyring=\"%s\" -o \"%s\" --batch --yes --always-trust --passphrase-fd=0 -d \"%s\"",
                GPGCOMMANDLINE, szGpgHome, szSecRing, szPubRing, szFilePlain, szFileEncrypted) < 0) {
    GPGEZDEBUG();
    return ERR_UNKNOWN;
  }

  gdp.lpdwStatus = lpdwStatus;
  gdp.szPassphrase = szPassphrase;

  *(gdp.lpdwStatus) = 0;
  dwErr = gpgez_dwRunGpg(szCommandLineOptions, _gpgez_dwProcDecrypt, (LPVOID)&gdp);

  if ((!(*lpdwStatus & STATUS_END_DECRYPTION)) || (*lpdwStatus & STATUS_DECRYPTION_FAILED)) {
    GPGEZDEBUG();
    dwErr = ERR_FAILED_TO_DECRYPT_FILE;
  } else if (!(*lpdwStatus & STATUS_VALIDSIGN)) {
    GPGEZDEBUG();
    dwErr = ERR_DECRYPTED_BUT_FAILED_TO_VALIDATE_SIGN;
  }
  return (dwErr);
}

/*------------------------------------------------------------*/
DWORD
_gpgez_dwProcDecrypt(struct childprocess *pcp, LPVOID pBuff) {
  struct gpgez_decrypt_params *pgdp;
  DWORD dwErr;
  CHAR szLine[MAX_STDLINE + 1];

  pgdp = (struct gpgez_decrypt_params *)pBuff;

  gpgez_bWritePipe(pcp->hStdIn, pgdp->szPassphrase);
  gpgez_bWritePipe(pcp->hStdIn, "\r\n");
  do {
    dwErr = gpgez_dwReadFileLine(pcp->hStdOut, szLine, MAX_STDLINE);
    if ((dwErr != ERR_NO_ERROR) && (dwErr != ERR_EOF) && (dwErr != ERR_READ_TOO_LONG_LINE)) break;

    if (gpgez_bIsGnupgMessage(szLine, GPGMSG_BAD_PASSPHRASE)) {
      GPGEZDEBUG();
      dwErr = ERR_BAD_PASSPHRASE;
      gpgez_dwProcStdOutDefault(pcp, NULL);
      break;
    }
    if (gpgez_bIsGnupgMessage(szLine, GPGMSG_VALIDSIG)) {
      *(pgdp->lpdwStatus) = *(pgdp->lpdwStatus) | STATUS_VALIDSIGN;
    }
    if (gpgez_bIsGnupgMessage(szLine, GPGMSG_DECRYPTION_FAILED)) {
      *(pgdp->lpdwStatus) = *(pgdp->lpdwStatus) | STATUS_DECRYPTION_FAILED;
    }
    if (gpgez_bIsGnupgMessage(szLine, GPGMSG_END_DECRYPTION)) {
      *(pgdp->lpdwStatus) = *(pgdp->lpdwStatus) | STATUS_END_DECRYPTION;
    }
    if (dwErr == ERR_EOF) {
      dwErr = ERR_NO_ERROR;
      break;
    }
  } while (TRUE);
  return (dwErr);
}

/*------------------------------------------------------------*/
