/* gpgez_genkey.c
 * Copyright (C) 2005 Kazuyoshi Kakihara
 *
 * This file is part of JanusDG.
 *
 * JanusDG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include "gpgez.h"
#include "gpgez_util.h"
#include "gpgez_rungpg.h"

#include "resource.h"

#define GPGMSG_KEY_CREATED "KEY_CREATED"
#define GPGMSG_KEY_NOT_CREATED "KEY_NOT_CREATED"

extern HINSTANCE g_hInstance;

struct gpgez_genkey_params {
  LPSTR szFpr;
  LPCSTR szGpgHome;
  LPCSTR szSecRing;
  LPCSTR szPubRing;
  LPCSTR szName;
  LPCSTR szEmail;
  LPCSTR szComment;
  LPCSTR szPassphrase;
  DWORD (*procProgress)(LPVOID);
  HWND hDlg;
};

DWORD _gpgez_dwProcGenkey(struct childprocess *pcp, LPVOID pBuff);

/*------------------------------------------------------------*/
DWORD
gpgez_genkey(LPCSTR szGpgHome, LPCSTR szSecRing, LPCSTR szPubRing, LPCSTR szName, LPCSTR szEmail, LPCSTR szComment, LPCSTR szPassphrase, LPSTR szFpr, DWORD (*procProgress)(LPVOID)) {
  CHAR szCommandLineOptions[MAX_PATH * 5];
  struct gpgez_genkey_params ggp;
  DWORD dwErr;

  if ((dwErr = gpgez_dwIsValidGpgHome(szGpgHome)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsInitializeSecRing(szSecRing)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsInitializePubRing(szPubRing)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsValidName(szName)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsValidEmail(szEmail)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsValidPassphrase(szPassphrase)) != ERR_NO_ERROR) return dwErr;

  if (_snprintf(szCommandLineOptions, sizeof(szCommandLineOptions), 
          "%s --homedir=\"%s\" --secret-keyring=\"%s\" --keyring=\"%s\" --batch --gen-key",
                GPGCOMMANDLINE, szGpgHome, szSecRing, szPubRing) < 0) {
    GPGEZDEBUG();
    return ERR_UNKNOWN;
  }

  ggp.szFpr = szFpr;
  ggp.szGpgHome = szGpgHome;
  ggp.szSecRing = szSecRing;
  ggp.szPubRing = szPubRing;
  ggp.szName = szName;
  ggp.szEmail = szEmail;
  ggp.szComment = szComment;
  ggp.szPassphrase = szPassphrase;
  ggp.procProgress = procProgress;

  dwErr = gpgez_dwRunGpg(szCommandLineOptions, _gpgez_dwProcGenkey, (LPVOID)&ggp);

  return (dwErr);
}

/*------------------------------------------------------------*/
DWORD
_gpgez_dwProcGenkey(struct childprocess *pcp, LPVOID pBuff) {
  struct gpgez_genkey_params *pggp;
  CHAR szLine[MAX_STDLINE + 1];
  DWORD dwErr;
  LPSTR szFpr;
  int iCnt = 0;
  BOOL bKeyCreated = FALSE;

  pggp = (struct gpgez_genkey_params *)pBuff;
  gpgez_bWritePipe(pcp->hStdIn, "Key-Type: DSA\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "Key-Length: 1024\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "Subkey-Type: ELG-E\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "Subkey-Length: 2048\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "Name-Real: "); gpgez_bWritePipe(pcp->hStdIn, pggp->szName); gpgez_bWritePipe(pcp->hStdIn, "\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "Name-Email: "); gpgez_bWritePipe(pcp->hStdIn, pggp->szEmail); gpgez_bWritePipe(pcp->hStdIn, "\r\n");
  if (strlen(pggp->szComment) > 0) {
    gpgez_bWritePipe(pcp->hStdIn, "Name-Comment: "); gpgez_bWritePipe(pcp->hStdIn, pggp->szComment); gpgez_bWritePipe(pcp->hStdIn, "\r\n");
  }
  gpgez_bWritePipe(pcp->hStdIn, "Expire-Date: 0\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "Passphrase: "); gpgez_bWritePipe(pcp->hStdIn, pggp->szPassphrase); gpgez_bWritePipe(pcp->hStdIn, "\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "%secring "); gpgez_bWritePipe(pcp->hStdIn, pggp->szSecRing); gpgez_bWritePipe(pcp->hStdIn, "\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "%pubring "); gpgez_bWritePipe(pcp->hStdIn, pggp->szPubRing); gpgez_bWritePipe(pcp->hStdIn, "\r\n");
  gpgez_bWritePipe(pcp->hStdIn, "%commit\r\n");
  CloseHandle(pcp->hStdIn);
  pcp->hStdIn = NULL;

  do {
    dwErr = gpgez_dwReadFileLine(pcp->hStdOut, szLine, MAX_STDLINE);
    if ((dwErr != ERR_NO_ERROR) && (dwErr != ERR_EOF) && (dwErr != ERR_READ_TOO_LONG_LINE)) break;
    if (pggp->procProgress) {
      iCnt++;
      if (iCnt == 100) {
        iCnt = 0;
      }
      pggp->procProgress(&iCnt);
    }
    if (gpgez_bIsGnupgMessage(szLine, GPGMSG_KEY_CREATED)) {
      bKeyCreated = TRUE;
      if (!(szFpr = strchr(szLine, ' '))) { GPGEZDEBUG(); dwErr = ERR_UNKNOWN; break; } szFpr++;
      if (!(szFpr = strchr(szFpr, ' '))) { GPGEZDEBUG(); dwErr = ERR_UNKNOWN; break; } szFpr++;
      if (!(szFpr = strchr(szFpr, ' '))) { GPGEZDEBUG(); dwErr = ERR_UNKNOWN; break; } szFpr++;
      strcpy(pggp->szFpr, szFpr);
    } else if (gpgez_bIsGnupgMessage(szLine, GPGMSG_KEY_NOT_CREATED)) {
      GPGEZDEBUG();
      dwErr = ERR_UNKNOWN;
      break;
    }
    if (dwErr == ERR_EOF) {
      dwErr = ERR_NO_ERROR;
      break;
    }
  } while (TRUE);
  if (!bKeyCreated) {
    GPGEZDEBUG();
    dwErr = ERR_FAILED_TO_CREATE_KEY;
  }
  return (dwErr);
}

/*------------------------------------------------------------*/
