/* gpgez_passwd.c
 * Copyright (C) 2005 Kazuyoshi Kakihara
 *
 * This file is part of JanusDG.
 *
 * JanusDG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include "gpgez.h"
#include "gpgez_util.h"
#include "gpgez_rungpg.h"

#define GPGMSG_NEED_PASSPHRASE "NEED_PASSPHRASE"
#define GPGMSG_NEED_PASSPHRASE_SYM "NEED_PASSPHRASE_SYM"
#define GPGMSG_GOOD_PASSPHRASE "GOOD_PASSPHRASE"
#define GPGMSG_BAD_PASSPHRASE "BAD_PASSPHRASE"
#define GPGMSG_GET_HIDDEN "GET_HIDDEN"
#define GPGMSG_GET_LINE "GET_LINE"

DWORD _gpgez_dwProcPasswd(struct childprocess *pcp, LPVOID pBuff);

struct passphrase_change {
  CHAR szOldPassphrase[MAX_PASSPHRASE + 1];
  CHAR szNewPassphrase[MAX_PASSPHRASE + 1];
};

/*------------------------------------------------------------*/
DWORD
gpgez_passwd(LPCSTR szGpgHome, LPCSTR szSecRing, LPCSTR szPubRing, LPCSTR szOldPassphrase, LPCSTR szNewPassphrase) {
  GPGEZ_KEYLIST KeyList;
  struct passphrase_change ppc;
  CHAR szFpr[MAX_FPR + 1];
  CHAR szCommandLineOptions[MAX_PATH * 6];
  DWORD dwErr;

  if ((dwErr = gpgez_dwIsValidGpgHome(szGpgHome)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsExistSecRing(szSecRing)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsExistPubRing(szPubRing)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsValidPassphrase(szOldPassphrase)) != ERR_NO_ERROR) return dwErr;
  if ((dwErr = gpgez_dwIsValidPassphrase(szNewPassphrase)) != ERR_NO_ERROR) return dwErr;

  KeyList.szFpr = NULL;
  KeyList.szFpr = NULL;
  KeyList.pNext = NULL;
  dwErr = gpgez_listseckeys(szGpgHome, szSecRing, &KeyList);
  if (KeyList.pNext == NULL) {
    GPGEZDEBUG();
    return ERR_ILLEGAL_SECRING_FORMAT;
  }
  if (dwErr != ERR_NO_ERROR) {
    gpgez_listkey_clear_node(&KeyList);
    return dwErr;
  }
  if ((KeyList.pNext)->pNext) {
    gpgez_listkey_clear_node(&KeyList);
    GPGEZDEBUG();
    return ERR_ILLEGAL_SECRING_FORMAT;
  }

  strcpy(ppc.szOldPassphrase, szOldPassphrase);
  strcpy(ppc.szNewPassphrase, szNewPassphrase);

  strcpy(szFpr, (KeyList.pNext)->szFpr);
  gpgez_listkey_clear_node(&KeyList);

  if (_snprintf(szCommandLineOptions, sizeof(szCommandLineOptions),
          "%s --homedir=\"%s\" --secret-keyring=\"%s\" --keyring=\"%s\" --no-default-keyring --command-fd=0 --edit-key %s passwd",
                GPGCOMMANDLINE, szGpgHome, szSecRing, szPubRing, szFpr) < 0) {
    GPGEZDEBUG();
    return ERR_UNKNOWN;
  }

  dwErr = gpgez_dwRunGpg(szCommandLineOptions, _gpgez_dwProcPasswd, (LPVOID)&ppc);

  return (dwErr);
}

#define STATUS_NEED_PASSPHRASE 1
#define STATUS_NEED_PASSPHRASE_SYM 2

/*------------------------------------------------------------*/
DWORD
_gpgez_dwProcPasswd(struct childprocess *pcp, LPVOID pBuff) {
  struct passphrase_change *pppc;
  DWORD dwErr;
  CHAR szLine[MAX_STDLINE + 1];
  BOOL bNewPass = FALSE;
  BOOL bSave = FALSE;

  pppc = (struct passphrase_change *)pBuff;
  do {
    dwErr = gpgez_dwReadFileLine(pcp->hStdOut, szLine, MAX_STDLINE);
    if ((dwErr != ERR_NO_ERROR) && (dwErr != ERR_EOF) && (dwErr != ERR_READ_TOO_LONG_LINE)) break;

    if (gpgez_bIsGnupgMessage(szLine, GPGMSG_NEED_PASSPHRASE)) {
      bNewPass = FALSE;
    } else if (gpgez_bIsGnupgMessage(szLine, GPGMSG_NEED_PASSPHRASE_SYM)) {
      bNewPass = TRUE;
    } else if (gpgez_bIsGnupgMessage(szLine, GPGMSG_GET_HIDDEN)) {
      if (!bNewPass) {
        gpgez_bWritePipe(pcp->hStdIn, pppc->szOldPassphrase);
        gpgez_bWritePipe(pcp->hStdIn, "\r\n");
      } else {
        gpgez_bWritePipe(pcp->hStdIn, pppc->szNewPassphrase);
        gpgez_bWritePipe(pcp->hStdIn, "\r\n");
      }
    } else if (gpgez_bIsGnupgMessage(szLine, GPGMSG_GOOD_PASSPHRASE)) {
      bSave = TRUE;
    } else if (gpgez_bIsGnupgMessage(szLine, GPGMSG_BAD_PASSPHRASE)) {
      bSave = FALSE;
    } else if (gpgez_bIsGnupgMessage(szLine, GPGMSG_GET_LINE)) {
      if (bSave) {
        gpgez_bWritePipe(pcp->hStdIn, "save\r\n");
      } else {
        gpgez_bWritePipe(pcp->hStdIn, "quit\r\n");
        GPGEZDEBUG();
        dwErr = ERR_BAD_PASSPHRASE;
        gpgez_dwProcStdOutDefault(pcp, NULL);
        break;
      }
    }
    if (dwErr == ERR_EOF) {
      dwErr = ERR_NO_ERROR;
      break;
    }
  } while (TRUE);
  return (dwErr);
}

/*------------------------------------------------------------*/
