/* gpgez_util.c
 * Copyright (C) 2005 Kazuyoshi Kakihara
 *
 * This file is part of JanusDG.
 *
 * JanusDG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include "gpgez_util.h"
#include "gpgez_error.h"
#include "resource.h"

#define GNUPG_MESSAGE_LABEL "[GNUPG:] "

extern HINSTANCE g_hInstance;

/*------------------------------------------------------------*/
/*
BOOL WINAPI
DllMain(HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved) {
  switch (fdwReason) {
  case DLL_PROCESS_ATTACH:
    g_hInstance = hinstDLL;
    break;
  case DLL_PROCESS_DETACH:
    break;
  default:
    break;
  }
  return TRUE;
}
*/
/*------------------------------------------------------------*/
VOID
gpgez_TrimString(LPSTR szStr) {
  LPSTR szSrc;
  LPSTR szDst;

  for (szSrc = szStr; *szSrc == ' '; szSrc++);
  if (szSrc != szStr) {
    for (szDst = szStr; *szSrc; szDst++, szStr++) *szDst = *szSrc;
    *szDst = '\0';
  }
  for (szSrc = szStr + strlen(szStr) - 1; (szSrc > szStr) && (*szSrc == ' '); szSrc--) *szSrc = '\0';
  return;
}

/*------------------------------------------------------------*/
PSTR
gpgez_strryen(PSTR pStr) {
  PSTR pPos = NULL, p;
  for (p = pStr; *p; p++) {
    if (*p == '\\') pPos = p;
    if (IsDBCSLeadByte(*p)) p++;
  }
  return pPos;
}

/*------------------------------------------------------------*/
BOOL
gpgez_bWritePipe(HANDLE hPipe, LPCSTR szBuff) {
  DWORD dwWritten;

  WriteFile(hPipe, szBuff, strlen(szBuff), &dwWritten, NULL);
  if (dwWritten != strlen(szBuff)) return FALSE;
  return TRUE;
}

/*------------------------------------------------------------*/
int CALLBACK
gpgez_DlgPasswordBoxProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam) {
  static PGPGEZ_PASSWORDBOXPARAM ppbParam;
  
  switch (uMsg) {
  case WM_INITDIALOG:
    ppbParam = (PGPGEZ_PASSWORDBOXPARAM)lParam;
    if (ppbParam->szTitle)
      SendMessage(hDlg, WM_SETTEXT, 0, (LPARAM)ppbParam->szTitle);
    if (ppbParam->szPrompt)
      SendMessage(GetDlgItem(hDlg, IDC_DIALOG_PASSWORD_PASSWORDPROMPT), WM_SETTEXT, 0, (LPARAM)ppbParam->szPrompt);
    SendMessage(GetDlgItem(hDlg, IDC_DIALOG_PASSWORD_EDITPASSWORD), EM_LIMITTEXT, ppbParam->dwLen, 0);
    SetFocus(GetDlgItem(hDlg, IDC_DIALOG_PASSWORD_EDITPASSWORD));
    return TRUE;
  case WM_COMMAND:
    switch (LOWORD(wParam)) {
    case IDOK:
      SendMessage(GetDlgItem(hDlg, IDC_DIALOG_PASSWORD_EDITPASSWORD), WM_GETTEXT, ppbParam->dwLen, (LPARAM)ppbParam->szBuff);
      EndDialog(hDlg, 1);
      return TRUE;
    case IDCANCEL:
      EndDialog(hDlg, 0);
      return TRUE;
    }
    break;
  default:
    return FALSE;
  }
  return FALSE;
}

/*------------------------------------------------------------*/
BOOL
gpgez_PasswordBox(HWND hwnd, LPSTR szBuff, DWORD dwLen, LPCSTR szPrompt, LPCSTR szTitle) {
  GPGEZ_PASSWORDBOXPARAM pbParam;
  INT nResult;

  strcpy(szBuff, "");
  pbParam.szBuff = szBuff;
  pbParam.dwLen = dwLen;
  pbParam.szPrompt = szPrompt;
  pbParam.szTitle = szTitle;
  nResult = DialogBoxParam(g_hInstance, MAKEINTRESOURCE(IDD_DIALOG_PASSWORD), hwnd, gpgez_DlgPasswordBoxProc, (LPARAM)&pbParam);
  if ((nResult == 0) || (nResult == -1)) {
    return FALSE;
  }
  return TRUE;
}

/*------------------------------------------------------------*/
BOOL
gpgez_bIsGnupgMessage(LPCSTR szBuff, LPCSTR szMessageLabel) {
  if (strncmp(szBuff, GNUPG_MESSAGE_LABEL, strlen(GNUPG_MESSAGE_LABEL))) return FALSE;
  if (strncmp(szBuff + strlen(GNUPG_MESSAGE_LABEL), szMessageLabel, strlen(szMessageLabel))) return FALSE;
  if (strchr(" \r\n\0", *(szBuff + strlen(GNUPG_MESSAGE_LABEL) + strlen(szMessageLabel)))) return TRUE;
  return FALSE;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwReadFileLine(HANDLE hPipe, LPSTR szBuff, DWORD dwMaxLen) {
  CHAR c;
  DWORD dwRead;
  DWORD dwLen;
  DWORD dwErr;

  dwLen = 0;
  strcpy(szBuff, "");
  while (TRUE) {
    if (!ReadFile(hPipe, &c, 1, &dwRead, NULL)) {
      if (GetLastError() != ERROR_BROKEN_PIPE) {
        GPGEZDEBUG();
        dwErr = ERR_UNKNOWN;
      } else {
        *(szBuff + dwLen) = '\0';
        dwErr = ERR_EOF;
      }
      break;
    }
    if (dwRead > 0) {
      if (c == '\n') {
        *(szBuff + dwLen) = '\0';
        if ((dwLen > 0) && (*(szBuff + dwLen - 1) == '\r')) {
          *(szBuff + dwLen - 1) = '\0';
        }
        dwErr = ERR_NO_ERROR;
        break;
      } else {
        if (dwLen < dwMaxLen) {
          *(szBuff + dwLen) = c;
        }
        dwLen++;
      }
    }
  }
  if (dwLen >= dwMaxLen) {
    GPGEZDEBUG();
    dwErr = ERR_READ_TOO_LONG_LINE;
  }
  return dwErr;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsValidGpgHome(LPCSTR szGpgHomeDir) {
  HANDLE hFile;
  WIN32_FIND_DATA FindFileData;

  hFile = FindFirstFile(szGpgHomeDir, &FindFileData);
  if (hFile == INVALID_HANDLE_VALUE) {
    GPGEZDEBUG();
    return ERR_FAILED_TO_OPEN_GPGHOME;
  } else {
    if (!(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
      FindClose(hFile);
      GPGEZDEBUG();
      return ERR_FAILED_TO_OPEN_GPGHOME;
    }
    FindClose(hFile);
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsExistSecRing(LPCSTR szSecring) {
  HANDLE hFile;

  hFile = CreateFile(szSecring, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    GPGEZDEBUG();
    return ERR_NOT_EXIST_SECRING_FILE;
  } else {
    CloseHandle(hFile);
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsExistPubRing(LPCSTR szPubring) {
  HANDLE hFile;

  hFile = CreateFile(szPubring, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    GPGEZDEBUG();
    return ERR_NOT_EXIST_PUBRING_FILE;
  } else {
    CloseHandle(hFile);
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsInitializeSecRing(LPCSTR szSecring) {
  HANDLE hFile;
  CHAR szBuff[MAX_STRING + 1];

  hFile = CreateFile(szSecring, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    hFile = CreateFile(szSecring, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);
    if (hFile == INVALID_HANDLE_VALUE) {
      GPGEZDEBUG();
      return ERR_FAILED_TO_CREATE_SECRING_FILE;
    }
    CloseHandle(hFile);
    DeleteFile(szSecring);
  } else {
    LoadString(g_hInstance, IDS_MESSAGE_OVERWRITE_SECRING_FILE, szBuff, MAX_STRING);
    if (MessageBox(NULL, szBuff, "CONFIRMATION", MB_OKCANCEL) == IDCANCEL) {
      CloseHandle(hFile);
      return ERR_USER_CANCEL;
    }
    CloseHandle(hFile);
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsInitializePubRing(LPCSTR szPubring) {
  HANDLE hFile;
  CHAR szBuff[MAX_PATH + 1];

  hFile = CreateFile(szPubring, GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    hFile = CreateFile(szPubring, GENERIC_WRITE, FILE_SHARE_WRITE, NULL, CREATE_NEW, FILE_ATTRIBUTE_NORMAL, NULL);
    if (hFile == INVALID_HANDLE_VALUE) {
      GPGEZDEBUG();
      return ERR_FAILED_TO_CREATE_PUBRING_FILE;
    }
    CloseHandle(hFile);
    DeleteFile(szPubring);
  } else {
    LoadString(g_hInstance, IDS_MESSAGE_OVERWRITE_PUBRING_FILE, szBuff, MAX_STRING);
    if (MessageBox(NULL, szBuff, "CONFIRMATION", MB_OKCANCEL) == IDCANCEL) {
      CloseHandle(hFile);
      return ERR_USER_CANCEL;
    }
    CloseHandle(hFile);
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsValidPassphrase(LPCSTR szPassphrase) {
  if (strlen(szPassphrase) < 8) {
    GPGEZDEBUG();
    return ERR_TOO_SHORT_PASSPHRASE;
  } else {
    return ERR_NO_ERROR;
  }
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsValidName(LPCSTR szName) {
  LPCSTR pSrc;

  if (strlen(szName) < 5) {
    GPGEZDEBUG();
    return ERR_INVALID_USER_NAME_TOO_SHORT;
  }
  if ((*szName >= '0') && (*szName <= '9')) {
    GPGEZDEBUG();
    return ERR_INVALID_USER_NAME_FIRST_CHARACTER_IS_NUMERIC;
  }
  for (pSrc = szName; *pSrc; pSrc++) {
    if (!isprint(*pSrc)) {
      GPGEZDEBUG();
      return ERR_INVALID_USER_NAME_ZENKAKU;
    }
    if ((*pSrc == '<') || (*pSrc == '>')) {
      GPGEZDEBUG();
      return ERR_INVALID_USER_NAME_ILLEGAL_CHARACTER;
    }
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_dwIsValidEmail(LPCSTR szEmail) {
  DWORD dwAt;
  LPCSTR pSrc;
  
  if ((*szEmail == '@') ||
    (*(szEmail + strlen(szEmail) - 1) == '@') ||
    (*(szEmail + strlen(szEmail) - 1) == '.') ||
    (strstr(szEmail, ".."))) {
    GPGEZDEBUG();
    return ERR_INVALID_EMAIL_ADDRESS_ILLEGAL_CHARACTER;
  }
  for (dwAt = 0, pSrc = szEmail; *pSrc; pSrc++) {
    if (*pSrc == '@') dwAt++;
  }
  if (dwAt != 1) {
    GPGEZDEBUG();
    return ERR_INVALID_EMAIL_ADDRESS_ILLEGAL_CHARACTER;
  }
  for (dwAt = 0, pSrc = szEmail; *pSrc; pSrc++) {
    if (*pSrc & 0x80) {
      GPGEZDEBUG();
      return ERR_INVALID_EMAIL_ADDRESS_ILLEGAL_CHARACTER;
    }
    if (*pSrc == '@') {
      dwAt++;
    } else {
      if ((dwAt == 0) &&
          (!isalnum(*pSrc)) &&
          (*pSrc != '+') &&
          (*pSrc != '-') &&
          (*pSrc != '_') &&
          (*pSrc != '.')) {
        GPGEZDEBUG();
        return ERR_INVALID_EMAIL_ADDRESS_ILLEGAL_CHARACTER;
      } else if ((dwAt == 1)  &&
          (!isalnum(*pSrc)) &&
          (*pSrc != '+') &&
          (*pSrc != '-') &&
          (*pSrc != '_') &&
          (*pSrc != '.')) {
        GPGEZDEBUG();
        return ERR_INVALID_EMAIL_ADDRESS_ILLEGAL_CHARACTER;
      }
    }
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_listkey_add_node(PGPGEZ_KEYLIST pKeyList, LPCSTR szUid, LPCSTR szFpr) {
  PGPGEZ_KEYLIST pKeyList_node;

  if ((pKeyList_node = (PGPGEZ_KEYLIST)malloc(sizeof(GPGEZ_KEYLIST))) == NULL) {
    GPGEZDEBUG();
    return ERR_FAILED_TO_ALLOCATE_MEMORY;
  }
  if ((pKeyList_node->szUid = strdup(szUid)) == NULL) {
    free(pKeyList_node);
    GPGEZDEBUG();
    return ERR_FAILED_TO_ALLOCATE_MEMORY;
  }
  if ((pKeyList_node->szFpr = strdup(szFpr)) == NULL) {
    free(pKeyList_node->szUid);
    free(pKeyList_node);
    GPGEZDEBUG();
    return ERR_FAILED_TO_ALLOCATE_MEMORY;
  }
  pKeyList_node->pNext = pKeyList->pNext;
  pKeyList->pNext = pKeyList_node;
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
gpgez_listkey_clear_node(PGPGEZ_KEYLIST pKeyList) {
  PGPGEZ_KEYLIST pKeyList_node;
  PGPGEZ_KEYLIST pKeyList_node_next;

  pKeyList_node = pKeyList->pNext;
  while (pKeyList_node) {
    pKeyList_node_next = pKeyList_node->pNext;
    if (pKeyList_node->szUid) free(pKeyList_node->szUid);
    if (pKeyList_node->szFpr) free(pKeyList_node->szFpr);
    free(pKeyList_node);
    pKeyList_node = pKeyList_node_next;
  }
  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
void
gpgez_ShowErrorMessage(HWND hwnd, DWORD dwErr) {
  CHAR szBuff[MAX_PATH + 1];
  DWORD dwIndex = 0;

  if (dwErr > ERR_WARN_MAX) {
    dwTranslateErrorCodeToString(dwErr, szBuff, MAX_PATH);
    MessageBox(hwnd, szBuff, "Message Box", MB_OK);
  }
  return;
}

/*------------------------------------------------------------*/
