/* gpgezw_main.c
 * Copyright (C) 2005 Kazuyoshi Kakihara
 *
 * This file is part of JanusDG.
 *
 * JanusDG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include "gpgezw.h"
#include <htmlhelp.h>
#include <stdio.h>

#pragma comment(lib, "comctl32.lib")

#define GPGCOMMAND "gpg.exe"
#define GPGBINDIR "bin"
#define GPGHOMEDIR "conf"
#define GPGSECRING "secring.gpg"
#define GPGPUBRING "pubring.gpg"
#define SUBKEY_GPGEZ "Software\\kzworks\\janusdg"
#define JANUSDG_OPTION_FILE "JanusDG.ini"

#define ID_TIMER 1
#define TIME_WAIT_TIMER 5000

#define NEED_INITIALIZE_TRUE 1
#define NEED_INITIALIZE_FALSE 0
#define NEED_INITIALIZE_ERROR 2

#include "resource.h"

HINSTANCE g_hInstance;
GPGEZPARAM g_gp;
CHAR g_szMessage[MAX_PATH + 1];
CHAR g_szHelpFileName[MAX_PATH + 1];
HWND g_hwndMain;
HWND g_hwndToolBar;
HWND g_hwndButton1;
HWND g_hwndButton2;
HWND g_hwndButton3;
HWND g_hwndButton4;
HWND g_hwndStatusBar;

void Cls_OnDestory(HWND);
void Cls_OnCommand(HWND, int, HWND, UINT);
BOOL Cls_OnCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct);

/*------------------------------------------------------------*/
INT
gpgezw_iGetProfileInt(LPSTR lpBuff, LPCSTR lpSection, LPCSTR lpName, INT iDefaultValue) {
  LPSTR pSrc, pDst, pSectionStart, pNameStart, pValueStart;
  CHAR cSave;
  INT iRet;

  pSrc = lpBuff;
  while (*pSrc) {
    pSectionStart = pSrc;
    while (isspace(*pSectionStart)) pSectionStart++;
    if (*pSectionStart != '[') {
      pSrc = strchr(pSectionStart, '\n');
      if (!pSrc) return iDefaultValue;
      pSrc++;
      continue;
    }
    pSectionStart++;
    pDst = strchr(pSectionStart, ']');
    if (!pDst) return iDefaultValue;
    cSave = *pDst; *pDst = '\0';
    if (strcmp(lpSection, pSectionStart)) {
      *pDst = cSave;
      pSrc = strchr(pSectionStart, '\n');
      if (!pSrc) return iDefaultValue;
      pSrc++;
      continue;
    }
    *pDst = cSave;
    pSrc = strchr(pSectionStart, '\n');
    if (!pSrc) return iDefaultValue;
    pSrc++;
    while (*pSrc) {
      while (isspace(*pSrc)) pSrc++;
      pNameStart = pSrc;
      pDst = pSrc;
      while ((*pDst != '=')&&(!isspace(*pDst))) pDst++;
      cSave = *pDst; *pDst = '\0';
      if (strcmp(lpName, pNameStart)) {
        *pDst = cSave;
        pSrc = strchr(pNameStart, '\n');
        if (!pSrc) return iDefaultValue;
        pSrc++;
        continue;
      }
      *pDst = cSave;
      while ((*pDst != '=')&&(*pDst != '\n') && (isspace(*pDst))) pDst++;
      if (*pDst == '\n') return iDefaultValue;
      pDst++;
      while ((*pDst != '\n') && (isspace(*pDst))) pDst++;
      if (*pDst == '\n') return iDefaultValue;
      pValueStart = pDst;
      while (isdigit(*pDst)) pDst++;
      if (!isspace(*pDst)) return iDefaultValue;
      cSave = *pDst; *pDst = '\0';
      iRet = atoi(pValueStart);
      *pDst = cSave;
      return iRet;
    }
  }
  return iDefaultValue;
}

/*------------------------------------------------------------*/
DWORD
_gpgezw_dwReadOption(VOID) {
  CHAR szFileName[MAX_PATH + 1];
  HANDLE hFile;
  BYTE szBuff[MAX_BUFF + 1];
  DWORD dwRead;
  DWORD dwErr;

  g_gp.fEncryptNoSign = FALSE;
  g_gp.fDecryptNoVerify = FALSE;

  /* IvVǂݍ */
  GetModuleFileName(NULL, szFileName, MAX_PATH);
  *(gpgez_strryen(szFileName) + 1) = '\0';
  strcat(szFileName, GPGHOMEDIR);
  strcat(szFileName, "\\");
  strcat(szFileName, JANUSDG_OPTION_FILE);
  hFile = CreateFile(szFileName, GENERIC_READ, FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE)
    return ERR_NO_ERROR; /* t@CȂΏ@\Lɂ */

  ZeroMemory(szBuff, MAX_BUFF);
  if (!ReadFile(hFile, szBuff, MAX_BUFF, &dwRead, NULL)) {
    GPGEZDEBUG();
    dwErr = ERR_FAILED_TO_READ_OPTION;
    goto finish_gpgezw_dwReadOption;
  }
  if (dwRead >= MAX_BUFF - 1) {
    GPGEZDEBUG();
    dwErr = ERR_UNKNOWN; /* t@CȂɑ傫͂Ȃ */
    goto finish_gpgezw_dwReadOption;
  }

  g_gp.fEncryptNoSign = gpgezw_iGetProfileInt(szBuff, "global", "encrypt_no_sign", 0) ? TRUE : FALSE;
  g_gp.fDecryptNoVerify = gpgezw_iGetProfileInt(szBuff, "global", "decrypt_no_verify", 0) ? TRUE : FALSE;
  dwErr = ERR_NO_ERROR;

finish_gpgezw_dwReadOption:
  CloseHandle(hFile);

  return dwErr;
}

/*------------------------------------------------------------*/
DWORD
_gpgezw_dwWriteOption(VOID) {
  CHAR szBuff[MAX_BUFF + 1], szFileName[MAX_PATH + 1];
  HANDLE hFile;
  DWORD dwWrite, dwWritten;

  dwWrite = _snprintf(szBuff, MAX_BUFF, "[global]\nencrypt_no_sign=%d\ndecrypt_no_verify=%d\n",
    (g_gp.fEncryptNoSign ? 1 : 0), (g_gp.fDecryptNoVerify ? 1 : 0));
  if (dwWrite < 0) {
    GPGEZDEBUG();
    return ERR_UNKNOWN;
  }

  GetModuleFileName(NULL, szFileName, MAX_PATH);
  *(gpgez_strryen(szFileName) + 1) = '\0';
  strcat(szFileName, GPGHOMEDIR);
  strcat(szFileName, "\\");
  strcat(szFileName, JANUSDG_OPTION_FILE);
  hFile = CreateFile(szFileName, GENERIC_WRITE, 0, NULL, OPEN_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE)
    return ERR_FAILED_TO_WRITE_OPTION;

  if (!WriteFile(hFile, szBuff, dwWrite, &dwWritten, NULL)) {
    CloseHandle(hFile);
    GPGEZDEBUG();
    return ERR_FAILED_TO_WRITE_OPTION;
  }
  if (dwWritten != dwWrite) {
    CloseHandle(hFile);
    GPGEZDEBUG();
    return ERR_FAILED_TO_WRITE_OPTION;
  }
  CloseHandle(hFile);

  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
VOID
_gpgezw_UpdateMenu(HWND hwnd) {
  HANDLE hMenu;
  MENUITEMINFO mii;

  hMenu = GetMenu(hwnd);
  mii.cbSize = sizeof(MENUITEMINFO);
  mii.fMask = MIIM_STATE;
  mii.fState = g_gp.fEncryptNoSign ? MFS_CHECKED : MFS_UNCHECKED;
  SetMenuItemInfo(hMenu, IDM_FLAG_ENCRYPT_NO_SIGN, FALSE, &mii);
  mii.fState = g_gp.fDecryptNoVerify ? MFS_CHECKED : MFS_UNCHECKED;
  SetMenuItemInfo(hMenu, IDM_FLAG_DECRYPT_NO_VERIFY, FALSE, &mii);
  return;
}

/*------------------------------------------------------------*/
DWORD
_gpgezw_dwGetGpgParam(PGPGEZPARAM pgp) {

  /* GPGHOME ̓oCiconfƌ߂ */
  GetModuleFileName(NULL, pgp->szGpgHome, MAX_PATH);
  *(gpgez_strryen(pgp->szGpgHome) + 1) = '\0';
  strcat(pgp->szGpgHome, GPGHOMEDIR);

  strcpy(pgp->szSecRing, pgp->szGpgHome);
  strcat(pgp->szSecRing, "\\");
  strcat(pgp->szSecRing, GPGSECRING);

  strcpy(pgp->szPubRing, pgp->szGpgHome);
  strcat(pgp->szPubRing, "\\");
  strcat(pgp->szPubRing, GPGPUBRING);

  return ERR_NO_ERROR;
}

/*------------------------------------------------------------*/
DWORD
_gpgezw_dwNeedInitialize(PGPGEZPARAM pgp) {
  WIN32_FIND_DATA FindFileData;
  HANDLE hFind;

  hFind = FindFirstFile(pgp->szGpgHome, &FindFileData);
  if (hFind == INVALID_HANDLE_VALUE) return NEED_INITIALIZE_TRUE;

  if (!(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
    FindClose(hFind);
    return NEED_INITIALIZE_ERROR;
  }
  
  FindClose(hFind);

  hFind = FindFirstFile(pgp->szSecRing, &FindFileData);
  if (hFind == INVALID_HANDLE_VALUE) return NEED_INITIALIZE_TRUE;
  FindClose(hFind);

  return NEED_INITIALIZE_FALSE;
}

/*------------------------------------------------------------*/
DWORD
_gpgezw_dwInitializeGpgKeys(HWND hwnd, PGPGEZPARAM pgp) {
  WIN32_FIND_DATA FindFileData;
  HANDLE hFind;
  DWORD dwErr;
  CHAR szPubkeyDir[MAX_PATH + 1];

  /* Ji[ptH_Ȃ΍쐬 */
  GetModuleFileName(NULL, szPubkeyDir, MAX_PATH);
  *(gpgez_strryen(szPubkeyDir) + 1) = '\0';
  strcat(szPubkeyDir, GPGEZW_PUBKEY_FOLDER);
  hFind = FindFirstFile(szPubkeyDir, &FindFileData);
  if (hFind == INVALID_HANDLE_VALUE) {
    if (!CreateDirectory(szPubkeyDir, NULL)) {
      GPGEZDEBUG();
      return ERR_ILLEGAL_INSTALLATION;
    }
  } else if (!(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
    FindClose(hFind);
    GPGEZDEBUG();
    return ERR_ILLEGAL_INSTALLATION;
  } else {
    FindClose(hFind);
  } 

  hFind = FindFirstFile(pgp->szGpgHome, &FindFileData);
  if (hFind == INVALID_HANDLE_VALUE) {
    if (!CreateDirectory(pgp->szGpgHome, NULL)) {
      GPGEZDEBUG();
      return ERR_ILLEGAL_INSTALLATION;
    }
  } else if (!(FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
    FindClose(hFind);
    GPGEZDEBUG();
    return ERR_ILLEGAL_INSTALLATION;
  } else {
    FindClose(hFind);
  }
  dwErr = gpgezw_dwInitialize(hwnd, pgp, NULL);

  return dwErr;
}

/*------------------------------------------------------------*/
DWORD
_gpgezw_dwTestPassphrase(HWND hwnd, PGPGEZPARAM pgp) {
  DWORD dwErr;
  DWORD dwStatus = 0;
  CHAR szWindowTitle[MAX_STRING + 1];
  CHAR szWindowPrompt[MAX_STRING + 1];
  CHAR szTempFolder[MAX_PATH + 1];
  CHAR szTempFileName[MAX_PATH + 1];
  
  /* ̌JɈÍāApXt[YmF */
  
  GetTempPath(MAX_PATH, szTempFolder);
  GetTempFileName(szTempFolder, "gpg", 0, szTempFileName);

  LoadString(g_hInstance, IDS_STRING_WSIGN_PASSPHRASE_WINDOW_TITLE, szWindowTitle, MAX_STRING);
  LoadString(g_hInstance, IDS_STRING_WSIGN_PASSPHRASE_WINDOW_PROMPT, szWindowPrompt, MAX_STRING);
  if (!gpgez_PasswordBox(hwnd, pgp->szPassphrase, MAX_PASSPHRASE, szWindowPrompt, szWindowTitle)) {
    dwErr = ERR_USER_CANCEL;
    return dwErr;
  }
  dwErr = gpgez_sign(pgp->szGpgHome, pgp->szSecRing, pgp->szPubRing, pgp->szPubRing, szTempFileName, pgp->szPassphrase, &dwStatus);
  DeleteFile(szTempFileName);

  return dwErr;
}

/*------------------------------------------------------------*/
HANDLE
_gpgezw_CreateToolBarButton(HWND hwnd, UINT uID, UINT uCnt) {
  CHAR szBuff[MAX_STRING + 1];
  HWND hwndChild;
  DWORD dwStyle;
  static int iButtonWidth;
  static int iButtonHeight;
  HDC hdc;
  TEXTMETRIC tm;
  RECT rc;

  LoadString(g_hInstance, uID, szBuff, MAX_STRING);
  dwStyle = WS_CHILD | WS_VISIBLE | BS_AUTORADIOBUTTON | BS_PUSHLIKE;
  if (uCnt == 0) dwStyle |= WS_GROUP;
  hwndChild = CreateWindowEx(0, "BUTTON", szBuff,
    dwStyle, uCnt * 60, 0, 60, 25,
    hwnd, (HMENU)uID, g_hInstance, NULL);
  SetWindowFont(hwndChild, GetStockObject(DEFAULT_GUI_FONT), TRUE);
  if (uCnt == 0) {
    hdc = GetDC(hwndChild);
    GetTextMetrics(hdc, &tm);
    iButtonWidth = tm.tmMaxCharWidth * 4;
    GetClientRect(hwnd, &rc);
    iButtonHeight = rc.bottom - rc.top;
    ReleaseDC(hwndChild, hdc);
  }
  MoveWindow(hwndChild, iButtonWidth * uCnt, 0, iButtonWidth, iButtonHeight, TRUE);
  return hwndChild;
}

/*------------------------------------------------------------*/
BOOL
Cls_OnCreate(HWND hwnd, LPCREATESTRUCT lpCreateStruct) {
  DWORD dwErr;
  DWORD dwInitialize;
  HDC hdc;
  TEXTMETRIC tm;
  GPGEZ_KEYLIST KeyList;
  CHAR szWindowTitle[MAX_STRING + MAX_UID + 1];
  CHAR szMessage[MAX_STRING + 1];

  g_hwndMain = hwnd;
  InitCommonControls();
  LoadString(g_hInstance, IDS_STRING_WMAIN_MESSAGE, g_szMessage, MAX_STRING);
  GetModuleFileName(NULL, g_szHelpFileName, MAX_PATH);
  *(gpgez_strryen(g_szHelpFileName) + 1) = '\0';
  strcat(g_szHelpFileName, "manual.chm");
//  strcat(g_szHelpFileName, "doc\\JanusDG\\manual\\index.html");

  /* c[o[p */
  SetWindowFont(hwnd, GetStockObject(DEFAULT_GUI_FONT), TRUE);
  hdc = GetDC(hwnd);
  GetTextMetrics(hdc, &tm);
  ReleaseDC(hwnd, hdc);
  g_hwndToolBar =  CreateWindowEx(WS_EX_TOOLWINDOW, TOOLBARCLASSNAME, NULL,
                                  WS_CHILD | WS_VISIBLE | TBSTYLE_AUTOSIZE, 0, 0, 0, tm.tmHeight * 12 / 10,
                                  hwnd, NULL, g_hInstance, NULL);

  g_hwndButton1 = _gpgezw_CreateToolBarButton(g_hwndToolBar, IDC_WLAUNCH_BUTTON1, 0);
  g_hwndButton2 = _gpgezw_CreateToolBarButton(g_hwndToolBar, IDC_WLAUNCH_BUTTON2, 1);
  g_hwndButton3 = _gpgezw_CreateToolBarButton(g_hwndToolBar, IDC_WLAUNCH_BUTTON3, 2);
  g_hwndButton4 = _gpgezw_CreateToolBarButton(g_hwndToolBar, IDC_WLAUNCH_BUTTON4, 3);

  Button_SetCheck(g_hwndButton1, BST_CHECKED);

  /* Xe[^Xo[p */
  g_hwndStatusBar = CreateWindowEx(0, STATUSCLASSNAME, NULL,
                                   WS_CHILD | SBARS_SIZEGRIP | CCS_BOTTOM | WS_VISIBLE, 0, 0, 0, 0,
                                   hwnd, NULL, g_hInstance, NULL);
  SendMessage(g_hwndStatusBar, SB_SIMPLE, TRUE, 0L);
  LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_ENCRYPT, szMessage, MAX_STRING);
  SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);

  DragAcceptFiles(hwnd, TRUE);

  dwErr = _gpgezw_dwGetGpgParam(&g_gp);
  if (dwErr != ERR_NO_ERROR) {
    gpgez_ShowErrorMessage(NULL, dwErr);
    return FALSE;
  }

  /* CEChE\ */
  ShowWindow(hwnd, SW_SHOW);

  /* szGpgHome݂ȂΏ߂Ă̋NȂ̂Ō */
  dwInitialize = _gpgezw_dwNeedInitialize(&g_gp);
  if (dwInitialize == NEED_INITIALIZE_ERROR) {
    GPGEZDEBUG();
    gpgez_ShowErrorMessage(NULL, ERR_FAILED_TO_OPEN_GPGHOME);
    return FALSE;
  } else if (dwInitialize == NEED_INITIALIZE_TRUE) {
    dwErr = _gpgezw_dwInitializeGpgKeys(hwnd, &g_gp);
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
      return FALSE;
    }
  } else {
    /* ݂΃pXt[Y̓ */
    dwErr = _gpgezw_dwTestPassphrase(hwnd, &g_gp);
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
      return FALSE;
    }
  }
  
  KeyList.szUid = NULL;
  KeyList.szFpr = NULL;
  KeyList.pNext = NULL;
  dwErr = gpgez_listseckeys(g_gp.szGpgHome, g_gp.szSecRing, &KeyList);
  if (KeyList.pNext == NULL) {
    gpgez_ShowErrorMessage(NULL, ERR_UNKNOWN);
    return FALSE;
  }
  if (dwErr != ERR_NO_ERROR) {
    gpgez_ShowErrorMessage(NULL, dwErr);
    gpgez_listkey_clear_node(&KeyList);
    return FALSE;
  }
  if ((KeyList.pNext)->pNext) {
    GPGEZDEBUG();
    gpgez_ShowErrorMessage(NULL, ERR_ILLEGAL_SECRING_FORMAT);
    gpgez_listkey_clear_node(&KeyList);
    return FALSE;
  }
  if (_snprintf(szWindowTitle, sizeof(szWindowTitle), "JanusDG - %s", (KeyList.pNext)->szUid) < 0) {
    GPGEZDEBUG();
    gpgez_ShowErrorMessage(NULL, ERR_UNKNOWN);
    gpgez_listkey_clear_node(&KeyList);
    return FALSE;
  }
  SendMessage(hwnd, WM_SETTEXT, 0, (LPARAM)szWindowTitle);
  gpgez_listkey_clear_node(&KeyList);

  dwErr = _gpgezw_dwReadOption();
  if (dwErr != ERR_NO_ERROR) {
    gpgez_ShowErrorMessage(NULL, dwErr);
    return FALSE;
  }
  _gpgezw_UpdateMenu(hwnd);

  return TRUE;
}

/*------------------------------------------------------------*/
void
Cls_OnDestroy(HWND hwnd) {
  DragAcceptFiles(hwnd, FALSE);
  PostQuitMessage(0);
  return;
}

/*------------------------------------------------------------*/
void
Cls_OnPaint(HWND hwnd) {
  HDC hdc;
  PAINTSTRUCT ps;
  RECT rc;
  SIZE sz;

  GetClientRect(hwnd, &rc);
  hdc = BeginPaint(hwnd, &ps);
  SelectObject(hdc, GetStockObject(DEFAULT_GUI_FONT));
  GetTextExtentPoint32(hdc, g_szMessage, strlen(g_szMessage), &sz);
  rc.top = ((rc.bottom - rc.top) - sz.cy) / 2;
  if (rc.top < 5) rc.top = 5;
  rc.left = ((rc.right - rc.left) - sz.cx) / 2;
  if (rc.left < 5) rc.left = 5;
  SetBkColor(hdc, 0x00ffffff);
  SetTextColor(hdc, RGB(0x7f, 0x7f, 0x7f));
  TextOut(hdc, rc.left, rc.top, g_szMessage, strlen(g_szMessage));
  EndPaint(hwnd, &ps);

  return;
}

/*------------------------------------------------------------*/
void
Cls_OnSize(HWND hwnd, UINT state, int cx, int cy) {
  SendMessage(g_hwndToolBar, WM_SIZE, 0L, MAKELPARAM((cx), (cy)));
  SendMessage(g_hwndStatusBar, WM_SIZE, 0L, 0L);
  return;
}

/*------------------------------------------------------------*/
LPSTR
szComposeFileNamesFromHDrop(HDROP hdrop, LPDWORD lpdwErr) {
  UINT uFileNo;
  UINT uCnt;
  LPSTR szFileNames;
  LPSTR pPos;
  CHAR szBasePath[MAX_PATH + 1];
  UINT uBasePathLen;
  CHAR szFileName[MAX_PATH + 1];
  UINT uFileNamesLen;

  uFileNo = DragQueryFile(hdrop, 0xFFFFFFFF, NULL, 0);

  /* t@C1̏ꍇ́At@CPƂŕԂ */
  if (uFileNo == 1) {
    DragQueryFile(hdrop, 0, szFileName, MAX_PATH);
    szFileNames = (LPSTR)malloc(strlen(szFileName) + 2);
    if (!szFileNames) {
      GPGEZDEBUG();
      *lpdwErr = ERR_FAILED_TO_ALLOCATE_MEMORY;
      return NULL;
    }
    strcpy(szFileNames, szFileName);
    pPos = strchr(szFileNames, '\0');

    /* ̏I[ɂ'\0't */
    *(pPos + 1) = '\0';
    return szFileNames;
  }

  /* ɂȂtH_𒲂ׂ */
  DragQueryFile(hdrop, 0, szBasePath, MAX_PATH);
  *(gpgez_strryen(szBasePath)) = '\0';

  uFileNamesLen = strlen(szBasePath) + 2;
  uBasePathLen = strlen(szBasePath) + 1;
  for (uCnt = 0; uCnt < uFileNo; uCnt++) {
    DragQueryFile(hdrop, uCnt, szFileName, MAX_PATH);
    uFileNamesLen += (strlen(szFileName) - uBasePathLen + 1);
  }
  szFileNames = (LPSTR)malloc(uFileNamesLen);
  if (!szFileNames) {
    GPGEZDEBUG();
    *lpdwErr = ERR_FAILED_TO_ALLOCATE_MEMORY;
    return NULL;
  }

  /* tH_[gtH_̏ꍇ́AtH_'\\'t */
  if (!*(szBasePath + 2)) strcat(szBasePath, "\\");

  strcpy(szFileNames, szBasePath);
  pPos = strchr(szFileNames, '\0') + 1;
  for (uCnt = 0; uCnt < uFileNo; uCnt++) {
    DragQueryFile(hdrop, uCnt, szFileName, MAX_PATH);
    strcpy(pPos, szFileName + uBasePathLen);
    pPos = strchr(pPos, '\0') + 1;
  }
  *pPos = '\0';
  return szFileNames;
}

/*------------------------------------------------------------*/
DWORD
gpgezw_launch_Encrypt(LPSTR szFileNames) {
  LPSTR pPos;
  WIN32_FIND_DATA FindFileData;
  HANDLE hFind;
  DWORD dwErr;
  CHAR szMessage[MAX_STRING + 1];
  
  if (szFileNames) {
    /* Ώۂ̃t@Cǂ */
    pPos = strchr(szFileNames, '\0') + 1;
    if (*pPos) {
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_MULTI_FILES;
    }
  
    /* ΏۂtH_łȂǂ */
    hFind = FindFirstFile(szFileNames, &FindFileData);
    if (hFind == INVALID_HANDLE_VALUE) {
      GPGEZDEBUG();
      return ERR_UNKNOWN;
    }
    if ((FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
      FindClose(hFind);
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_TO_PROCESS_FOLDER;
    }
    FindClose(hFind);
  }
  
  LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_ENCRYPT, szMessage, MAX_STRING);
  SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  
  dwErr = gpgezw_dwEncrypt(g_hwndMain, &g_gp, szFileNames);
  
  if (dwErr == ERR_NO_ERROR) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_END_ENCRYPT, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  }
  return dwErr;
}

/*------------------------------------------------------------*/
DWORD
gpgezw_launch_Decrypt(LPSTR szFileNames) {
  LPSTR pPos;
  WIN32_FIND_DATA FindFileData;
  HANDLE hFind;
  DWORD dwErr;
  CHAR szMessage[MAX_STRING + 1];
  
  if (szFileNames) {
    /* Ώۂ̃t@Cǂ */
    pPos = strchr(szFileNames, '\0') + 1;
    if (*pPos) {
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_MULTI_FILES;
    }
    
    /* ΏۂtH_łȂǂ */
    hFind = FindFirstFile(szFileNames, &FindFileData);
    if (hFind == INVALID_HANDLE_VALUE) {
      GPGEZDEBUG();
      return ERR_UNKNOWN;
    }
    if ((FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
      FindClose(hFind);
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_TO_PROCESS_FOLDER;
    }
    FindClose(hFind);
  }
  
  LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_DECRYPT, szMessage, MAX_STRING);
  SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  
  dwErr = gpgezw_dwDecrypt(g_hwndMain, &g_gp, szFileNames);
  
  if (dwErr == ERR_NO_ERROR) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_END_DECRYPT, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  }
  return dwErr;
}

/*------------------------------------------------------------*/
DWORD
gpgezw_launch_Sign(LPSTR szFileNames) {
  LPSTR pPos;
  WIN32_FIND_DATA FindFileData;
  HANDLE hFind;
  DWORD dwErr;
  CHAR szMessage[MAX_STRING + 1];
  
  if (szFileNames) {
    /* Ώۂ̃t@Cǂ */
    pPos = strchr(szFileNames, '\0') + 1;
    if (*pPos) {
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_MULTI_FILES;
    }
    
    /* ΏۂtH_łȂǂ */
    hFind = FindFirstFile(szFileNames, &FindFileData);
    if (hFind == INVALID_HANDLE_VALUE) {
     GPGEZDEBUG();
      return ERR_UNKNOWN;
    }
    if ((FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
      FindClose(hFind);
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_TO_PROCESS_FOLDER;
    }
    FindClose(hFind);
  }
  
  LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_SIGN, szMessage, MAX_STRING);
  SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  
  dwErr = gpgezw_dwSign(g_hwndMain, &g_gp, szFileNames);
  
  if (dwErr == ERR_NO_ERROR) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_END_SIGN, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  }
  
  return dwErr;
}

/*------------------------------------------------------------*/
DWORD
gpgezw_launch_Verify(LPSTR szFileNames) {
  LPSTR pPos;
  WIN32_FIND_DATA FindFileData;
  HANDLE hFind;
  DWORD dwErr;
  CHAR szMessage[MAX_STRING + 1];
  
  if (szFileNames) {
    /* Ώۂ̃t@Cǂ */
    pPos = strchr(szFileNames, '\0') + 1;
    if (*pPos) {
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_MULTI_FILES;
    }
    
    /* ΏۂtH_łȂǂ */
    hFind = FindFirstFile(szFileNames, &FindFileData);
    if (hFind == INVALID_HANDLE_VALUE) {
      GPGEZDEBUG();
      return ERR_UNKNOWN;
    }
    if ((FindFileData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) {
      FindClose(hFind);
      GPGEZDEBUG();
      return ERR_NOT_ALLOWED_TO_PROCESS_FOLDER;
    }
    FindClose(hFind);
  }
  
  LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_VERIFY, szMessage, MAX_STRING);
  SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  
  dwErr = gpgezw_dwVerify(g_hwndMain, &g_gp, szFileNames);
  
  if (dwErr == ERR_NO_ERROR) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_VALID_SIGN, szMessage, MAX_STRING);
    MessageBox(g_hwndMain, szMessage, "Message Box", MB_OK);
    LoadString(g_hInstance, IDS_STRING_WMAIN_END_VERIFY, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
  }

  return dwErr;
}

/*------------------------------------------------------------*/
DWORD
gpgezw_launch_Passphrase(VOID) {
  DWORD dwErr;
  CHAR szMessage[MAX_STRING + 1];

  dwErr = gpgezw_dwPassphrase(g_hwndMain, &g_gp, "");
  if (dwErr == ERR_NO_ERROR) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_PASSPHRASE_CHANGED, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)(LPSTR)"pXt[YύX܂");
  }
  return dwErr;
}

/*------------------------------------------------------------*/
void
Cls_OnDropFiles(HWND hwnd, HDROP hdrop) {
  DWORD dwErr = ERR_NO_ERROR;
  LPSTR szFileNames;

  SetWindowPos(hwnd, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE | SWP_NOSIZE | SWP_SHOWWINDOW);
  szFileNames = szComposeFileNamesFromHDrop(hdrop, &dwErr);

  if (Button_GetCheck(g_hwndButton1) == BST_CHECKED) { /* Í */
    dwErr = gpgezw_launch_Encrypt(szFileNames);
  } else if (Button_GetCheck(g_hwndButton2) == BST_CHECKED) { /*  */
    dwErr = gpgezw_launch_Decrypt(szFileNames);
  } else if (Button_GetCheck(g_hwndButton3) == BST_CHECKED) { /*  */
    dwErr = gpgezw_launch_Sign(szFileNames);
  } else if (Button_GetCheck(g_hwndButton4) == BST_CHECKED) { /*  */
    dwErr = gpgezw_launch_Verify(szFileNames);
  }
  if (dwErr != ERR_NO_ERROR) {
    gpgez_ShowErrorMessage(g_hwndMain, dwErr);
  }
  SetTimer(hwnd, ID_TIMER, TIME_WAIT_TIMER, NULL);

  free(szFileNames);
  return;
}

/*------------------------------------------------------------*/
int CALLBACK
DlgVersionInfoProc(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam) {
  switch (uMsg) {
  case WM_INITDIALOG:
    return TRUE;
  case WM_COMMAND:
    switch (LOWORD(wParam)) {
    case IDOK:
      EndDialog(hDlg, 1);
      return TRUE;
    }
    break;
  default:
    return FALSE;
  }
  return FALSE;
}

/*------------------------------------------------------------*/
void
Cls_OnCommand(HWND hwnd, int id, HWND hwndCtl, UINT codeNotify) {
  DWORD dwErr;
  DWORD dwOption;
  CHAR szMessage[MAX_STRING + 1];

  switch(id) {
  case IDM_ENCRYPT:
    dwErr = gpgezw_launch_Encrypt(NULL);
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(g_hwndMain, dwErr);
    }
    SetTimer(hwnd, ID_TIMER, TIME_WAIT_TIMER, NULL);
    break;
  case IDM_SIGN:
    dwErr = gpgezw_launch_Sign(NULL);
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(g_hwndMain, dwErr);
    }
    SetTimer(hwnd, ID_TIMER, TIME_WAIT_TIMER, NULL);
    break;
  case IDM_DECRYPT:
    dwErr = gpgezw_launch_Decrypt(NULL);
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(g_hwndMain, dwErr);
    }
    SetTimer(hwnd, ID_TIMER, TIME_WAIT_TIMER, NULL);
    break;
  case IDM_VERIFY:
    dwErr = gpgezw_launch_Verify(NULL);
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(g_hwndMain, dwErr);
    }
    SetTimer(hwnd, ID_TIMER, TIME_WAIT_TIMER, NULL);
    break;
  case IDM_VIEWKEY:
    dwErr = gpgezw_dwViewKey(hwnd, &g_gp, "");
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
    }
    break;
  case IDM_FINGERPRINT:
    dwErr = gpgezw_dwShowMyFpr(hwnd, &g_gp, "");
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
    }
    break;
  case IDM_EXPORT:
    dwErr = gpgezw_dwExportMyKey(hwnd, &g_gp, "");
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
    }
    break;
  case IDM_PASSPHRASE:
    dwErr = gpgezw_launch_Passphrase();
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
    } else {
    }
    break;
/*
  case IDM_INIT:
    dwErr = gpgezw_dwInitialize(hwnd, &g_gp, "");
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
      SendMessage(hwnd, WM_CLOSE, 0, 0L);
    }
    break;
*/
  case IDC_WLAUNCH_BUTTON1:
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_ENCRYPT, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
    break;
  case IDC_WLAUNCH_BUTTON2:
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_DECRYPT, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
    break;
  case IDC_WLAUNCH_BUTTON3:
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_SIGN, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
    break;
  case IDC_WLAUNCH_BUTTON4:
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_VERIFY, szMessage, MAX_STRING);
    SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
    break;
  case IDM_FLAG_ENCRYPT_NO_SIGN:
    g_gp.fEncryptNoSign = g_gp.fEncryptNoSign ? FALSE : TRUE;
    dwOption = 0;
    if (g_gp.fEncryptNoSign) dwOption = dwOption | GPGEZW_OPTION_ENCRYPT_NO_SIGN;
    if (g_gp.fDecryptNoVerify) dwOption = dwOption | GPGEZW_OPTION_DECRYPT_NO_VERIFY;
    dwErr = _gpgezw_dwWriteOption();
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
    }
    _gpgezw_UpdateMenu(hwnd);
    break;
  case IDM_FLAG_DECRYPT_NO_VERIFY:
    g_gp.fDecryptNoVerify = g_gp.fDecryptNoVerify ? FALSE : TRUE;
    dwOption = 0;
    if (g_gp.fEncryptNoSign) dwOption = dwOption | GPGEZW_OPTION_ENCRYPT_NO_SIGN;
    if (g_gp.fDecryptNoVerify) dwOption = dwOption | GPGEZW_OPTION_DECRYPT_NO_VERIFY;
    dwErr = _gpgezw_dwWriteOption();
    if (dwErr != ERR_NO_ERROR) {
      gpgez_ShowErrorMessage(NULL, dwErr);
    }
    _gpgezw_UpdateMenu(hwnd);
    break;
  case IDM_HELP:
    HtmlHelp(hwnd, g_szHelpFileName, HH_DISPLAY_TOC, 0);
/*    ShellExecute(NULL, NULL, g_szHelpFileName, NULL, NULL, SW_SHOWNORMAL); */
    break;
  case IDM_VERSION:
    DialogBoxParam(g_hInstance, MAKEINTRESOURCE(IDD_DIALOG_VERSIONINFO), hwnd, DlgVersionInfoProc, (LPARAM)0L);
    break;
  case IDM_EXIT:
    SendMessage(hwnd, WM_CLOSE, 0, 0L);
    break;
  }
}

/*------------------------------------------------------------*/
void
Cls_OnTimer(HWND hwnd, UINT id) {
  CHAR szMessage[MAX_STRING + 1];

  KillTimer(hwnd, ID_TIMER);
  if (Button_GetCheck(g_hwndButton1)) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_ENCRYPT, szMessage, MAX_STRING);
  } else if (Button_GetCheck(g_hwndButton2)) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_DECRYPT, szMessage, MAX_STRING);
  } else if (Button_GetCheck(g_hwndButton3)) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_SIGN, szMessage, MAX_STRING);
  } else if (Button_GetCheck(g_hwndButton4)) {
    LoadString(g_hInstance, IDS_STRING_WMAIN_BEGIN_VERIFY, szMessage, MAX_STRING);
  }
  SendMessage(g_hwndStatusBar, SB_SETTEXT, 255 | 0, (WPARAM)szMessage);
}

/*------------------------------------------------------------*/
void
Cls_OnClose(HWND hwnd) {
  KillTimer(hwnd, ID_TIMER);
  DestroyWindow(hwnd);
}

/*------------------------------------------------------------*/
LRESULT CALLBACK
procWnd(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam) {
  switch (uMsg) {
    HANDLE_MSG(hwnd, WM_CREATE, Cls_OnCreate);
    HANDLE_MSG(hwnd, WM_DESTROY, Cls_OnDestroy);
    HANDLE_MSG(hwnd, WM_COMMAND, Cls_OnCommand);
    HANDLE_MSG(hwnd, WM_PAINT, Cls_OnPaint); 
    HANDLE_MSG(hwnd, WM_SIZE, Cls_OnSize);
    HANDLE_MSG(hwnd, WM_DROPFILES, Cls_OnDropFiles);
    HANDLE_MSG(hwnd, WM_TIMER, Cls_OnTimer);
    HANDLE_MSG(hwnd, WM_CLOSE, Cls_OnClose);
  default:
    return (DefWindowProc(hwnd, uMsg, wParam, lParam));
  }
  return 0L;
}

/*------------------------------------------------------------*/
int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow)
{
  MSG msg;
  WNDCLASSEX wc;
  BOOL bRet;
  DWORD dwCookie = 0;
  static CHAR szMainWindowClassName[] = "JANUSDG";

  g_hInstance = hInstance;

  HtmlHelp(NULL, NULL, HH_INITIALIZE, (DWORD)&dwCookie);

  ZeroMemory(&wc, sizeof(WNDCLASSEX));
  wc.cbSize = sizeof(WNDCLASSEX);
  wc.style = CS_HREDRAW | CS_VREDRAW;
  wc.lpfnWndProc = procWnd;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = 0;
  wc.hInstance = hInstance;
  wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON1));
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = (HBRUSH) GetStockObject(WHITE_BRUSH);
  wc.lpszMenuName = MAKEINTRESOURCE(IDR_MENU1);
  wc.lpszClassName = szMainWindowClassName;
  wc.hIconSm = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON1));
  RegisterClassEx(&wc);

  msg.hwnd = CreateWindowEx(
    WS_EX_ACCEPTFILES , szMainWindowClassName, "JanusDG", WS_OVERLAPPEDWINDOW,
    CW_USEDEFAULT, CW_USEDEFAULT, 320, 240,
    NULL, NULL, hInstance, NULL);

  ShowWindow(msg.hwnd, nCmdShow);
  UpdateWindow(msg.hwnd);

  while ((bRet = GetMessage(&msg, NULL, 0, 0)) != 0) {
    if (bRet == -1) {
      MessageBox(msg.hwnd, "GetMessage Error", "Error", MB_OK);
      break;
    } else {
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  }
  HtmlHelp(NULL, NULL, HH_UNINITIALIZE, (DWORD)dwCookie);
  return (int)msg.wParam;
}

/*------------------------------------------------------------*/

