/***************************************************************************
    Copyright          : (C) 2002 by Neoworks Limited. All rights reserved
    URL                : http://www.neoworks.com
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * Track.java
 *
 * Created on 26 March 2002, 18:24
 */

package com.neoworks.jukex;

import java.util.Collection;
import java.net.URL;
import java.util.Date;

/**
 * The Track class encapsulates all information about a single music track.
 *
 * @author Nick Vincent (<a href="mailto:nick@neoworks.com">nick@neoworks.com</a>)
 */
public interface Track
{	
	/**
	 * Retrieve all of the AttributeValue objects corresponding to the given 
	 * Attribute.
	 *
	 * This, for instance, could allow you to get all of the artists who performed
	 * on a Track by passing an Attribute object representing the 'performer' attribute.
	 *
	 * @param attribute The Attribute for which the values are to be retrieved
	 * @return A Collection of AttributeValue objects
	 */
	public Collection getAttributeValues(Attribute attribute);
	
	/**
	 * Retrieve a Collection of all Attribute objects which this track has.
	 *
	 * @return A Collection of all Attribute objects which this track has.
	 */
	public Collection getKnownAttributes();
	
	/**
	 * Removes all of the values for this track from the passed Attribute
	 *
	 * @param attribute The Attribute to remove the values of
	 */
	public void clearAttribute(Attribute attribute);
	
	/**
	 * Replace all of the current attribute values with the one passed
	 *
	 * @param attribute The Attribute to have its values changed
	 * @param value The AttributeValue to become the new value
	 */
	public void replaceAttributeValues(Attribute attribute, AttributeValue value);
	
	/**
	 * Adds an AttributeValue to an existing Attribute on this Track.
	 *
	 * @param attribute The Attribute to whom the value is to be added
	 * @param value The AttributeValue to add
	 */
	public void addAttributeValue(Attribute attribute, AttributeValue value);
	
	/**
	 * Return the URL of the Track's location
	 *
	 * @return The URL where the Track can be found
	 */
	public URL getURL();
	
	/**
	 * Return the database ID of this Track
	 *
	 * @return The database ID of this Track
	 */
	public long getId();
	
	/**
	 * Return the last time that this Track was updated in the database.
	 *
	 * Note that this is only triggered by setUpdatedDate, and not by any alterations
	 * made to Attributes/AttributeValues.
	 *
	 * @return The last time the track was updated.
	 */
	public Date getUpdatedDate();
	
	/**
	 * Set the time that this Track was updated.
	 *
	 * This takes a parameter to avoid time problems where the clocks on many
	 * systems maybe slightly out of sync and mean that suddenly all tracks are
	 * updated in the future.
	 *
	 * @param updatedDate The updated Date to attach to the track.
	 */
	public void setUpdatedDate( Date updatedDate );
	
	/**
	 * Gets the first AttributeValue belonging to the passed attribute
	 * @param attribute The Attribute whose values we are retrieving
	 */
	public AttributeValue getAttributeValue( Attribute attribute );
	
	/**
	 * Gets the first AttributeValue belonging to the Attribute referred to by
	 * the passed String.
	 *
	 * This is a convenience method in order that code may be more concise.
	 *
	 * @param attributename The name of the Attribute whose first value is to be 
	 *                      retrieved.
	 */
	public AttributeValue getAttributeValue( String attributename );
}
