/***************************************************************************
    Copyright          : (C) 2002 by Neoworks Limited. All rights reserved
    URL                : http://www.neoworks.com
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * TrackStore.java
 *
 * Created on 27 March 2002, 14:19
 */

package com.neoworks.jukex;

import java.net.*;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.LinkedList;
import java.util.Set;
import java.sql.*;

import com.neoworks.jukex.sqlimpl.*;
import com.neoworks.jukex.Playlist;

import com.neoworks.connectionpool.PoolManager;

import org.apache.log4j.Category;

/**
 * TrackStore is the core of the JukeX system.
 *
 * The TrackStore provides the means to access all core objects.  It handles all
 * database operations.
 *
 * @author Nick Vincent (<a href="mailto:nick@neoworks.com">nick@neoworks.com</a>)
 */
public interface TrackStore
{
	/**
	 * Get a Track object by URL
	 *
	 * @param url The URL of the track
	 * @return The Track object
	 */
	public Track getTrack( URL url );

	/**
	 * Get a Track object by database id
	 *
	 * @param id The track id
	 * @return The Track object
	 */
	public Track getTrack( long id );

	/**
	 * Retrieve a <code>List</code> of <code>Track</code> objects with ids as
	 * specified.
	 *
	 * @param ids Array of ids for the Tracks to retrieve
	 * @return A <code>List</code> of <code>Track</code> objects
	 */
	public List getTracks( long [] ids );

	/**
	 * Get the number of tracks in the Track database
	 *
	 * @return The number of tracks
	 */
	public long getTrackCount();

	/**
	 * Get an array containing the ids of the tracks in the Track database
	 *
	 * @return An array of track ids
	 */
	public long[] getTrackIds();

	/**
	 * Retrieve a <code>List</code> of <code>Track</code> objects with ids as
	 * specified in a List of Integers.
	 *
	 * @param ids The List of Integer objects of ids of Tracks to retrieve
	 * @return A <code>List</code> of <code>Track</code> objects
	 */
	public List getTracks( List ids );
        
	/**
	 * Create a track in the database with the specified url and return the 
	 * object representing that track.
	 *
	 * @param url The URL of the track to be added to the database
	 * @param modifieddate The date that this Track entry was modified
	 * @return The newly added Track
	 */
	public Track storeTrack( URL url , Date modifieddate );
        
	/**
	 * Return the Attribute object representing the attribute with that name
	 *
	 * @param The name of the Attribute to retrieve
	 * @return The <code>Attribute</code> object representing the name passed, or <code>null</code>, if it is not found.
	 */
	public Attribute getAttribute(String name);
        
	/**
	 * Get a list of all the attributes currently supported by the system
	 *
	 * @return The <code>Set</code> of all <code>Attribute</code> objects 
	 * currently available in the database.
	 */
	public Set getAttributes();
	
	/**
	 * Return an attribute of the corresponding name and type, creating a new
	 * attribute within the system if necessary.
	 *
	 * @return An attribute with the corresponding name and type.
	 */
	public Attribute createAttribute( String name , int type ) throws Exception;
	
	/**
	 * Return all of the Playlists currently defined
	 *
	 * @return A Collection of Playlist objects currently known by the TrackStore
	 */
	public Collection getAllPlaylists();
	
	/**
	 * Get the Playlist with the specified name
	 *
	 * @return A Playlist with the specified name, or <code>null</code> if one
	 *         does not exist.
	 */
	public Playlist getPlaylist( String name );
	
	/**
	 * Get the Playlist with the specified name, or create it if it does not
	 * already exist.
	 *
	 * @return A Playlist with the specified name
	 */
	public Playlist createPlaylist( String name );	
	
	/**
	 * Return a batch track loader
	 *
	 * @return A BatchTrackLoader from this TrackStore
	 */
	public BatchTrackLoader getBatchTrackLoader();
}
