/***************************************************************************
    Copyright          : (C) 2002 by Neoworks Limited. All rights reserved
    URL                : http://www.neoworks.com
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * InMemoryPlaylist.java
 *
 * Created on 22 May 2002, 11:25
 */

package com.neoworks.jukex.sqlimpl;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import java.sql.Connection;

import com.neoworks.jukex.Playlist;
import com.neoworks.jukex.Track;
import com.neoworks.jukex.tracksource.TrackSourcePipeline;
import com.neoworks.jukex.tracksource.TrackSourcePipelineElement;

/**
 * A Playlist implementation that is not persisted to the database
 *
 * @author Nick Vincent (<a href="mailto:nick@neoworks.com">nick@neoworks.com</a>)
 */
public class InMemoryPlaylist extends LinkedList implements Playlist
{
	private String name;
	private boolean enabled;

	/**
	 * Creates a new instance of InMemoryPlaylist
	 */
	public InMemoryPlaylist()
	{
		name = "Transient Playlist";
	}

	/**
	 * Creates a new instance of InMemoryPlaylist
	 *
	 * @param name The name of this playlist
	 */
	public InMemoryPlaylist(String name)
	{
		this.name = name;
	}

	/**
	 * Get the name of this TrackSource
	 *
	 * @return A String identifying the TrackSource
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * Is this playlist enabled
	 */
	public boolean isEnabled()
	{
		return enabled;
	}

	public void enable()
	{
		enabled = true;
	}

	public void disable()
	{
		enabled = false;
	}

	/**
	 * Get the next track from the source.
	 *
	 * @return A Track object
	 */
	public Track getNextTrack()
	{
		if ( super.size() > 0 )
		{
			return ( (Track) super.remove( 0 ) );
		}
		else
		{
			return null;
		}
	}
	
	/**
	 * Remove a track from the playlist
	 *
	 * @param index The index of the track to remove
	 */
	public boolean removeTrack(int index)
	{
		super.remove(index);
		return true;
	}

	/**
	 * Get the next TrackSourcePipelineElement.
	 * Does nothing in this implementation as it isn't supported.
	 *
	 * @return <code>null</code>
	 */
	public TrackSourcePipelineElement getNextTrackSource()
	{
		return null;
	}

	/**
	 * Get the current state of this PipelineElement for persisting. As this
	 * will never be persisted it just returns null.
	 *
	 * @return null
	 */
	public Map getState()
	{
		return null;
	}

	/**
	 * Get the Track at the specified index
	 *
	 * @param index The index of the Track to get
	 * @return The Track at the specified index
	 */
	public Track getTrack(int index)
	{
		Track retval = (Track) super.remove( index );
		return retval;
	}
	
	/**
	 * Peek at the next count tracks from this TrackSourcePipelineElement
	 *
	 * @param count The number of Tracks to peek ahead at
	 * @return A List of Track objects
	 */
	public List peekTracks(int count)
	{
		int max = (count > super.size()) ? super.size() : count ;
		return super.subList( 0 , max );
	}
	
	/**
	 * Tells this tracksource of a new link in the chain
	 *
	 * @return The previous TrackSource
	 */
	public TrackSourcePipelineElement setNextTrackSource(TrackSourcePipelineElement next)
	{
		return null;
	}
	
	/**
	 * Set which TrackSourcePipeline this PipelineElement is currently a part of
	 *
	 * @param pipe The TrackSourcePipeline to which you now belong
	 * @return The PipelineElement, to allow for chaining of calls
	 */
	public TrackSourcePipelineElement setOwner(TrackSourcePipeline pipe)
	{
		return null;
	}
	
	/**
	 * Load the configuration of this PipelineElement. This does nothing as the
	 * configuration is never stored.
	 *
	 * @param state a Map of values keyed by Strings representing the state of this PipelineElement.
	 * @return success
	 */
	public boolean setState(Map state)
	{
		return false;
	}
	
	/**
	 * Persist the current state of the PipelineElement into the database
	 *
	 * @param conn The database connection to use, which must have AutoCommit disabled
	 * @return success
	 */
	public boolean storeState(Connection conn)
	{
		return false;
	}
	
	/**
	 * Return a String summarising the configuration of the task the source is 
	 * performing
	 *
	 * @return A String
	 */
	public String getSummary()
	{
		return "InMemoryPipeline: This holds the personal playlist for a user";
	}
	
	/**
	 * Return a String describing what, in general terms, this TrackSource does
	 *
	 * @return A String
	 */
	public String getDescription()
	{
		return "Currently holding the playlist for a user";
	}

	/**
	 * Set the name of this Playlist
	 *
	 * @param name The new name
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	public Object clone()
	{
		InMemoryPlaylist retVal = (InMemoryPlaylist)super.clone();
		retVal.name = this.name;
		retVal.enabled = this.enabled;
		return retVal;
	}
}
