/***************************************************************************
    Copyright          : (C) 2002 by Neoworks Limited. All rights reserved
    URL                : http://www.neoworks.com
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * JukeXAttribute.java
 *
 * Created on 10 April 2002, 12:00
 */

package com.neoworks.jukex.sqlimpl;

import com.neoworks.jukex.*;
import java.util.Locale;
import java.util.ResourceBundle;

import org.apache.log4j.Category;

/**
 * SQL implementation of Attribute for JukeX.
 *
 * @author Nick Vincent (<a href="mailto:nick@neoworks.com">nick@neoworks.com</a>)
 */
public class JukeXAttribute implements Attribute, DatabaseObject, Comparable
{
	private static final Category log = Category.getInstance(JukeXAttribute.class.getName());
	private static final boolean logDebugEnabled = log.isDebugEnabled();
	private static final boolean logInfoEnabled = log.isInfoEnabled();

	private String _name = null;
	private long _id = -1;
	private int _type = 0;
	
	/**
	 * Package private constructor
	 *
	 * @param id The database id of this Attribute
	 * @param name The name of this Attribute
	 * @param type The type of this Attribute
	 */
	JukeXAttribute( int id , String name , int type )
	{
		_name = name;
		_id = id;
		_type = type;
	}

	/**
	 * Return the database ID of this object
	 * 
	 * @return The database ID of this object
	 */
	public long getId()
	{
		return _id;
	}

	/**
	 * Get a localised name for this attribute
	 *
	 * @param locale The required Locale for the name
	 * @return A localised name for the attribute
	 */
	public String getLocalisedName(Locale locale)
	{
		String retVal = null;
		ResourceBundle bundle = ResourceBundle.getBundle("AttributeResources",locale,Thread.currentThread().getContextClassLoader());
		try {
			if (logDebugEnabled)log.debug("Getting localised name for " + _name + " with locale " + locale);
			retVal = bundle.getString(_name);
		} catch (Exception e) {
			log.warn("Exception encountered while attempting to get localised name for attribute " + _name, e);
			retVal = _name;
		}
		return retVal;
	}

	/**
	 * Get the name of this attribute
	 *
	 * @return The name of the attribute
	 */
	public String getName()
	{
		return _name;
	}

	/**
	 * Compare this Attribute to another Object.
	 *
	 * @return a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object.
	 */
	public int compareTo(Object obj)
	{
		if ( ( obj instanceof Attribute ) && ( obj instanceof DatabaseObject ) )
		{
			return ( new Long(_id) ).compareTo( new Long( ((DatabaseObject)obj).getId() ) );
		}
		else
		{
			return -1;
		}
	}

	/**
	 * Indicate whether an Object is equal to this Attribute.
	 *
	 * @return Success
	 */
	public boolean equals(Object obj)
	{
		if (obj instanceof JukeXAttribute)
		{
			return ( _id == ((DatabaseObject)obj).getId() );
		}
		else
		{
			return false;
		}
	}

	/**
	 * Get a hashcode for this Attribute
	 *
	 * @return An integer hashcode
	 */
	public int hashCode()
	{
		return _name.hashCode();
	}

	/**
	 * Get a String representation of this Attribute
	 *
	 * @return A String
	 */
	public String toString()
	{
		return ( "[" +_id + ":" + _name + ":" + _type + "]" );
	}	

	/**
	 * Get an AttributeValue object representing the integer that was 
	 * passed
	 *
	 * @param val The integer to retrieve an AttributeValue object for
	 * @return An AttributeValue representing the passed integer.
	 */
	public AttributeValue getAttributeValue(int val)
	{
		return new JukeXAttributeValue( this , val );
	}

	/**
	 * Get an AttributeValue object representing the String object that was 
	 * passed
	 *
	 * @param s The String to retrieve an AttributeValue object for
	 * @return An AttributeValue representing the passed String.
	 */
	public AttributeValue getAttributeValue(String s)
	{
		if ( s == null )
		{
			return null;
		}
		else
		{
			return new JukeXAttributeValue( this , s );
		}
	}

	/**
	 * Return the type of the attribute, which can be equal to Attribute.TYPE_STRING or
	 * Attribute.TYPE_INT
	 *
	 * @return The type of the Attribute
	 */
	public int getType()
	{
		return _type;
	}
}
