/***************************************************************************
    Copyright          : (C) 2002 by Neoworks Limited. All rights reserved
    URL                : http://www.neoworks.com
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*
 * TrackSourcePipelineElementSkeleton.java
 *
 * Created on 02 May 2002, 14:48
 */

package com.neoworks.jukex.tracksource;

import com.neoworks.jukex.*;
import com.neoworks.util.SizeLimitedLinkedHashMap;
import com.neoworks.util.CollectionUtils;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.Iterator;

/**
 * PipelineElement that guarantees a minimum number of tracks played between repeats
 *
 * @author Nigel Atkinson <a href="mailto:nigel@neoworks.com">nigel@neoworks.com</a>
 */
public class NoRepeatPipelineElement extends TrackSourcePipelineElementSkeleton
{
	private SizeLimitedLinkedHashMap LUT;

	/**
	 * Public constructor
	 *
	 * @param n Minimum number of tracks played between repeats
	 */
	public NoRepeatPipelineElement()
	{
		super("NoRepeat");
		LUT = new SizeLimitedLinkedHashMap(10);
	}


	/**
	 * Public constructor
	 *
	 * @param n Minimum number of tracks played between repeats
	 */
	public NoRepeatPipelineElement(int n)
	{
		super("NoRepeat");
		LUT = new SizeLimitedLinkedHashMap(n);
	}

	/**
	 * Get the interval (in tracks played) in which tracks must not be repeated
	 *
	 * @return The interval
	 */
	public int getInterval()
	{
		return LUT.getMaxSize();
	}

	/**
	 * Set the interval (in tracks played) in which tracks must not be repeated
	 *
	 * @param size The interval size
	 */
	public void setInterval(int size)
	{
		LUT.setMaxSize(size);
	}

	/**
	 * Get the current state of this PipelineElement for persisting
	 *
	 * @return a String keyed Map of objects representing the state of 
	 * this PipelineElement, e.g. for a Playlist this would be information
	 * allowing the reconstruction of the current playlist queue.
	 */
	public Map getState()
	{
		Map retVal = new HashMap();
		retVal.put("name",name);
		retVal.put("LUT",LUT);
		retVal.put("enabled",new Boolean(isEnabled()));
		return retVal;
	}

	/**
	 * Load the configuration of this PipelineElement
	 *
	 * @param state a Map of values keyed by Strings representing the state of this PipelineElement.
	 * @return success
	 */
	public boolean setState( Map state )
	{
		LUT = (SizeLimitedLinkedHashMap)state.get("LUT");
		if (((Boolean)state.get("enabled")).booleanValue())
		{
			enable();
		} else {
			disable();
		}
		return true;
	}

	/**
	 * Get a track at random
	 *
	 * @return a Track object
	 */
	public Track getNextTrack()
	{
		Track t = null;

		if (isEnabled())
		{
			Long l = null;
			
			do {
				t = delegateGetNextTrack();
				if (t == null)
				{
					break;
				}
				l = new Long(t.getId());
			} while (LUT.containsKey(l));
			if (l != null)
			{
				LUT.put(l,l);
			}
		} else {
			t = delegateGetNextTrack();
		}
		return t;
	}

	/**
	 * Peek up the track list
	 *
	 * @param count The number of tracks to peek
	 * @return A List of Track objects
	 */
	public List peekTracks(int count)
	{
		List retVal = null;
		Iterator i = null;
		int peekCount = count;
		int peekListSize = 0;
		boolean notEnoughTracksToPeek = false;

		if (isEnabled())
		{
			do {
				retVal = delegatePeekTracks(peekCount);
				peekListSize = retVal.size();
				if (peekListSize != peekCount)
				{
					notEnoughTracksToPeek = true;
				}
				peekCount = count;
				retVal = CollectionUtils.distinctList(retVal);
				peekCount += (peekListSize - retVal.size());
				i = retVal.iterator();
				while (i.hasNext())
				{
					if (LUT.containsKey(new Long(((Track)i.next()).getId())))
					{
						i.remove();
						peekCount++;
					}
				}
				if (notEnoughTracksToPeek)
				{
					break;
				}
			} while (retVal.size() < count);
		} else {
			retVal = delegatePeekTracks(count);
		}
		return retVal;
	}

	/**
	 * Clone this FilterPipelineElement
	 *
	 * @return A clone of this RandomiserPipelineElement
	 */
	public Object clone()
	{
		return new NoRepeatPipelineElement(LUT.getMaxSize());
	}
	
	/**
	 * Return a String describing what, in general terms, this TrackSource does
	 *
	 * @return A String
	 */
	public String getDescription()
	{
		return "NoRepeatPipelinElement: Stops the same track being played twice within a specified period";
	}
	
	/**
	 * Return a String summarising the configuration of the task the source is
	 * performing
	 *
	 * @return A String
	 */
	public String getSummary()
	{
		if ( super.isEnabled() )
		{
			return "Preventing repeats during any "+this.LUT.getMaxSize()+" track period";
		}
		else
		{
			return "Disabled";
		}
	}
	
}
