/***************************************************************************
    Copyright          : (C) 2002 by Neoworks Limited. All rights reserved
    URL                : http://www.neoworks.com
 ***************************************************************************/
/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

package com.neoworks.shout;

import java.net.*;
import java.lang.*;
import java.io.*;
import java.util.*;
import org.apache.log4j.Category;

/**
 * This class holds the information for a [Shout|Ice]Cast stream. The information
 * is read from a configuration file.
 *
 * @author Nigel Atkinson (<a href="mailto:nigel@neoworks.com">nigel@neoworks.com</a>)
 */
public class Stream {
	private static final Category log = Category.getInstance(Stream.class.getName());
	private static final boolean logDebugEnabled = log.isDebugEnabled();
	private static final boolean logInfoEnabled = log.isInfoEnabled();

	private static File configFile = null;

	private String name = "";
	private String description = "";
	private String url = "";
	private String server = "";
	private String password = "";
	private String publicShoutStream = "";
	private String remotepass = "";
	private int bitrate = 0;
	private int port = 0;
	private int trackInfoPort = 0;

	/**
	 * Public constructor
	 */
	public Stream()
	{
	}

	/**
	 * Is this stream public?
	 *
	 * @return "yes" if the stream is public, "no" otherwise
	 */
	public String getPublicShout()
	{
		return this.publicShoutStream;
	}

	/**
	 * Set the public stream flag
	 *
	 * @param val The new value ("yes"|"no")
	 */
	public void setPublicShout(String val)
	{
		this.publicShoutStream = val;
	}

	/**
	 * Get the stream name
	 *
	 * @return The name
	 */
	public String getStreamName()
	{
		return this.name;
	}

	/**
	 * Set the stream name
	 *
	 * @param val The new name
	 */
	public void setStreamName(String val)
	{
		this.name = val;
	}

	/**
	 * get the stream description
	 *
	 * @return The description
	 */
	public String getStreamDescription()
	{
		return this.description;
	}

	/**
	 * Set the stream description
	 *
	 * @param val The new description
	 */
	public void setStreamDescription(String val)
	{
		this.description = val;
	}

	/**
	 * Get the stream URL
	 *
	 * @return The URL
	 */
	public String getURL()
	{
		return this.url;
	}

	/**
	 * Set the stream URL
	 *
	 * @param val The new URL
	 */
	public void setURL(String val)
	{
		this.url = val;
	}

	/**
	 * Get the server name
	 *
	 * @return The server name
	 */
	public String getServerName()
	{
		return this.server;
	}

	/**
	 * Set the server name
	 *
	 * @param val The new server name
	 */
	public void setServerName(String val)
	{
		this.server = val;
	}

	/**
	 * Get the server password
	 *
	 * @return The server password
	 */
	public String getPassword()
	{
		return this.password;
	}

	/**
	 * Set the server password
	 *
	 * @param val The new password
	 */
	public void setPassword(String val)
	{
		this.password = val;
	}

	/**
	 * Get the stream bitrate indicator
	 *
	 * @return The bitrate (kb/s)
	 */
	public int getBitrate()
	{
		return this.bitrate;
	}

	/**
	 * Set the stream bitrate indicator
	 *
	 * @param val The new bitrate (kb/s)
	 */
	public void setBitrate(int val)
	{
		this.bitrate = val;
	}

	/**
	 * Get the port to send track information to
	 *
	 * @return The port number
	 */
	public int getTrackInfoPort()
	{
		return this.trackInfoPort;
	}

	/**
	 * Set the port to send track information to
	 *
	 * @param val The new port number
	 */
	public void setTrackInfoPort(int val)
	{
		this.trackInfoPort = val;
	}

	/**
	 * Get the source connection port
	 *
	 * @return The port number
	 */
	public int getPort()
	{
		return this.port;
	}

	/**
	 * Set the source connection port
	 *
	 * @param val The new port number
	 */
	public void setPort(int val)
	{
		this.port = val;
	}

	/**
	 * Load the config file and check it
	 *
	 * @param filename The name of the config file
	 * @return Success
	 * @exception FileNotFoundException If the config file cannot be found
	 */
	public boolean loadConfig(String filename) throws FileNotFoundException
	{
		configFile = new File(filename);
		return checkConfig();
	}

	/**
	 * Read the config file
	 *
	 * @return Success
	 */
	private boolean checkConfig() 
	{
		if (configFile == null)
		{
			return false;
		}
		String configToken = "";
		String configLine = "";
		String value = "";
		try {
			BufferedReader configReader = new BufferedReader(new InputStreamReader(new FileInputStream(configFile)));
			// While we have more lines in the file, run a check on each ignoring
			// those which start with #.
			while ( ( configLine = configReader.readLine() ) != null )
			{
				if ( configLine.startsWith("#") == false && configLine.equals("") == false )
				{ 
					StringTokenizer st = new StringTokenizer(configLine, " ");
					configToken = st.nextToken();
					value = configLine.substring(configToken.length() + 1).trim();
					if ( configToken.equals("name") ) {
						name = value;
					} else if ( configToken.equals("description") ) {
						description = value;
					} else if ( configToken.equals("url") ) {
						url = value;
					} else if ( configToken.equals("bitrate") ) {
						bitrate = Integer.parseInt(value);
					} else if ( configToken.equals("public") ) {
						publicShoutStream = ( value.equals("yes") )?"yes":"no";
					} else if ( configToken.equals("server") ) {
						server = value;
					} else if ( configToken.equals("trackinfoport") ) {
						trackInfoPort = Integer.parseInt(value);
					} else if ( configToken.equals("port") ) {
						port = Integer.parseInt(value);
					} else if ( configToken.equals("password") ) {
						password = value;
					}
				}
			}
		} catch (Exception e) {
			log.warn("Exception encountered while reading config file",e);
			return false;
		}

		return true;
	}
}
